/*
 * CurveExactMovementInfo.h
 *
 * Created 9/2/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/2/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Contains information for an exact curve movement.
  */
 
 #ifndef CURVE_EXACT_MOVEMENT_INFO_H
 #define CURVE_EXACT_MOVEMENT_INFO_H
 
 template <typename T> class CurveExactMovementInfo;
 
 #include "global.h"
 
 #include "CurveMovementInfo.h"
 #include "Vector.h"
 
 /**
  * Class specification for CurveExactMovementInfo
  */
 template <typename T>
 class CurveExactMovementInfo : public CurveMovementInfo<T>
 {
 // Data Members
 private:
    // 0 <= t <= 1
    // P(t) = at^2 + vt + p
    VECTOR3_TYPE _begin_point;
    
 // Local Functions
 public:
    CurveExactMovementInfo( const VECTOR3_TYPE& acceleration = VECTOR3_TYPE( ZERO, ZERO, ZERO ), 
                            const VECTOR3_TYPE& velocity = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                            const VECTOR3_TYPE& begin_point = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                            const T distance = ONE, const double duration = 0.0 );
    CurveExactMovementInfo( const CurveExactMovementInfo<T>& curve_exact_movement_info );
    virtual ~CurveExactMovementInfo();
    inline CurveExactMovementInfo<T>& operator=( const CurveExactMovementInfo<T>& curve_exact_movement_info );
    inline const VECTOR3_TYPE& get_begin_point() const;
    inline void set_begin_point( const VECTOR3_TYPE& begin_point );
    virtual inline double process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  * P(t) = (distance*(acceleration*t^2 + velocity*t)) + p
  */
 template <typename T>
 CurveExactMovementInfo<T>::CurveExactMovementInfo( const VECTOR3_TYPE& acceleration, 
                                                    const VECTOR3_TYPE& velocity,
                                                    const VECTOR3_TYPE& begin_point,
                                                    const T distance, const double duration )
                           : CurveMovementInfo<T>( acceleration, velocity, distance, duration ),
                             _begin_point( begin_point ) // _begin_point( begin_point + (distance*Vector::get_normal<T>( begin_point )) )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CurveExactMovementInfo<T>::CurveExactMovementInfo( const CurveExactMovementInfo<T>& curve_exact_movement_info )
                      : CurveMovementInfo<T>( curve_exact_movement_info ),
                        _begin_point( curve_exact_movement_info._begin_point )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CurveExactMovementInfo<T>::~CurveExactMovementInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CurveExactMovementInfo to this CurveExactMovementInfo.
  *
  * @param (const CurveExactMovementInfo<T>&) curve_exact_movement_info
  * @return CurveExactMovementInfo<T>&
  */
 template <typename T>
 inline CurveExactMovementInfo<T>& CurveExactMovementInfo<T>::operator=( const CurveExactMovementInfo<T>& curve_exact_movement_info )
 {
    CurveMovementInfo<T>::operator=( curve_exact_movement_info );
    _begin_point = curve_exact_movement_info._begin_point;
    
    return *this;
 }
 
 /**
  * get_begin_point() returns the curve movement starting location.
  *
  * @return const VECTOR3_TYPE&
  */
 template <typename T>
 inline const VECTOR3_TYPE& CurveExactMovementInfo<T>::get_begin_point() const
 {
    return _begin_point;
 }
 
 /**
  * set_begin_point() sets the curve movement starting location to the specified begin point.
  *
  * @param (const VECTOR3_TYPE&) begin_point
  */
 template <typename T>
 inline void CurveExactMovementInfo<T>::set_begin_point( const VECTOR3_TYPE& begin_point )
 {
    _begin_point = begin_point;
 }
 
 /**
  * process_movement() processes the movement specified by this MovementInfo, given the Object to process on,
  * the MoveInfo, and current time. The duration unprocessed returned is greater than zero if this MovementInfo
  * has finished processing, and there is time leftover.
  *
  * @param (Object<T>*) obj_Ptr
  * @param (MoveInfo<T>*) move_info_Ptr
  * @param (double) current_time
  * @return double - the time left unprocessed (i.e. a double greater than zero if this MovementInfo is done processing)
  */
 template <typename T>
 inline double CurveExactMovementInfo<T>::process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time )
 {  
    nassertr( obj_Ptr != NULL, MovementInfo<T>::get_duration() );
    
    double move_action_invoke_time( move_info_Ptr->get_time_move_action_was_invoked() );
    double last_process_time( move_info_Ptr->get_time_last_processed_move_task() );
    nassertr( last_process_time >= move_action_invoke_time, MovementInfo<T>::get_duration() );
    
    double duration( MovementInfo<T>::get_duration() );
    double elapse_time_since_invocation( current_time - move_action_invoke_time );
    
    if ( elapse_time_since_invocation > 0.0 && (CurveMovementInfo<T>::get_acceleration() != VECTOR3_TYPE(ZERO,ZERO,ZERO) || CurveMovementInfo<T>::get_velocity() != VECTOR3_TYPE(ZERO,ZERO,ZERO)) )
    {
        // gradually move for only the movement duration (measured in seconds)
        //if ( elapse_time_since_invocation > duration ) // if true, this will be the last time we process this movement_info
        
        double invocation_ratio( (elapse_time_since_invocation >= duration ? 1.0 : elapse_time_since_invocation / duration) );
        double invocation_ratio_squared( invocation_ratio * invocation_ratio );
        
        // move the Object to a specific location
        // (a*invocation_ratio^2 + v*invocation_ratio) + p
        obj_Ptr->set_pos( (CurveMovementInfo<T>::get_acceleration()*invocation_ratio_squared) + (CurveMovementInfo<T>::get_velocity()*invocation_ratio) + _begin_point );
    }
    
    if ( elapse_time_since_invocation > duration )
        return elapse_time_since_invocation - duration;
    else // elapse_time_since_invocation <= duration
        return 0.0;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // CURVE_EXACT_MOVEMENT_INFO_H